<?php
/**
 * Plugin Name: NetPayBD
 * Plugin URI: https://netpaybd.com/documentation
 * Description: Integrates NetPayBD Payment Gateway with WooCommerce.
 * Version: 1.0.0
 * Author: NetPayBD
 * Author URI: https://netpaybd.com
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: netpaybd-for-woocommerce
 * WC requires at least: 3.0.0
 * WC tested up to: 8.2.0
 */

if (!defined('ABSPATH')) {
    exit;
}

add_action('plugins_loaded', 'init_netpaybd_gateway_class');

function init_netpaybd_gateway_class() {
    if (!class_exists('WC_Payment_Gateway')) {
        return;
    }

    class WC_Gateway_NetPayBD extends WC_Payment_Gateway {

        public function __construct() {
            $this->id = 'netpaybd';
            $this->icon = ''; 
            $this->has_fields = false;
            $this->method_title = 'NetPayBD Gateway';
            $this->method_description = 'Allows payments with NetPayBD Payment Gateway.';

            $this->init_form_fields();
            $this->init_settings();

            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');
            $this->enabled = $this->get_option('enabled');

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            add_action('woocommerce_api_wc_gateway_netpaybd', array($this, 'handle_callback'));
        }

        public function init_form_fields() {
            $this->form_fields = array(
                'enabled' => array(
                    'title'   => 'Enable/Disable',
                    'type'    => 'checkbox',
                    'label'   => 'Enable NetPayBD Gateway',
                    'default' => 'yes'
                ),
                'title' => array(
                    'title'       => 'Title',
                    'type'        => 'text',
                    'description' => 'This controls the title which the user sees during checkout.',
                    'default'     => 'Pay with NetPayBD',
                    'desc_tip'    => true,
                ),
                'description' => array(
                    'title'       => 'Description',
                    'type'        => 'textarea',
                    'description' => 'This controls the description which the user sees during checkout.',
                    'default'     => 'Pay securely via bKash, Nagad, Rocket, and more.',
                ),
                'username' => array(
                    'title'       => 'Merchant Username',
                    'type'        => 'text',
                    'description' => 'Enter your NetPayBD merchant username.',
                ),
                'api_key' => array(
                    'title'       => 'API Key',
                    'type'        => 'text',
                    'description' => 'Enter your NetPayBD API Key.',
                )
            );
        }

        public function process_payment($order_id) {
            global $woocommerce;
            $order = wc_get_order($order_id);

            $gateway_username = $this->get_option('username');
            $gateway_api_key = $this->get_option('api_key');
            $amount = $order->get_total();

            $success_url = $this->get_return_url($order);
            
            $callback_url = WC()->api_request_url('WC_Gateway_NetPayBD');

            if (empty($gateway_username) || empty($gateway_api_key)) {
                wc_add_notice('Payment gateway is not configured properly. Please contact support.', 'error');
                return;
            }

            $payment_params = [
                'username'      => $gateway_username,
                'api'           => $gateway_api_key,
                'amount'        => $amount,
                'success_url'   => $success_url,
                'callback_url'  => $callback_url . '?order_id=' . $order_id,
            ];
            
            $gateway_url = 'https://pay.netpaybd.com/?' . http_build_query($payment_params);

            $order->update_status('on-hold', __('Awaiting payment via NetPayBD.', 'netpaybd-for-woocommerce'));
            
            return array(
                'result'   => 'success',
                'redirect' => $gateway_url
            );
        }

        public function handle_callback() {
            if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['order_id'])) {
                
                $order_id = absint($_GET['order_id']);
                $order = wc_get_order($order_id);

                if (!$order) {
                    wp_die('Invalid Order ID', 'NetPayBD Callback Error', array('response' => 400));
                    exit;
                }
                
                $callback_data = $_POST;
                $status = isset($callback_data['status']) ? sanitize_text_field($callback_data['status']) : '';
                $transaction_id = isset($callback_data['transaction_id']) ? sanitize_text_field($callback_data['transaction_id']) : '';

                if ($status === 'success') {
                    $order->payment_complete($transaction_id);
                    $order->add_order_note(
                        sprintf(
                            'NetPayBD payment successful. Transaction ID: %s',
                            $transaction_id
                        )
                    );
                    
                    wc_reduce_stock_levels($order_id);
                } else {
                    $order->update_status('failed', sprintf('NetPayBD payment failed. Transaction ID: %s', $transaction_id));
                }

                echo "Callback received successfully.";
                exit;
            }
            
            wp_die('Invalid callback request', 'NetPayBD Callback Error', array('response' => 400));
        }
    }
}

function add_netpaybd_gateway_class($methods) {
    $methods[] = 'WC_Gateway_NetPayBD';
    return $methods;
}

add_filter('woocommerce_payment_gateways', 'add_netpaybd_gateway_class');